// Basic class implementation for access to USB HID devices
//
// (C) 2001 Copyright Cleware GmbH
// All rights reserved
//
// History:
// 05.01.01	ws	Initial coding
// 02.11.01	ws	ported to Linux

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include "USBaccess.h"
#include "USBaccessBasic.h"


CUSBaccess::CUSBaccess() {
	X = new CUSBaccessBasic ;
	}

CUSBaccess::~CUSBaccess() {
	delete X ;
	}


// returns number of found Cleware devices
int
CUSBaccess::OpenCleware() {
	int rval = X->OpenCleware() ;

	return rval ;
	}

// return true if ok, else false
int
CUSBaccess::CloseCleware() {
	int rval = X->CloseCleware() ;

	return rval ;
	}

HANDLE
CUSBaccess::GetHandle(int deviceNo) { 
	return X->GetHandle(deviceNo) ; 
	}

int 
CUSBaccess::GetVersion(int deviceNo) { 
	return X->GetVersion(deviceNo) ; 
	}

int 
CUSBaccess::GetUSBType(int deviceNo) { 
	return X->GetUSBType(deviceNo) ; 
	}

int	 
CUSBaccess::GetSerialNumber(int deviceNo) { 
	return X->GetSerialNumber(deviceNo) ; 
	}



// returns 1 if ok or 0 in case of an error
int		
CUSBaccess::GetValue(int deviceNo, unsigned char *buf, int bufsize) {
	int rval = X->GetValue(deviceNo, 65441, 3, buf, bufsize) ;

	return rval ;
	}


int 
CUSBaccess::SetValue(int deviceNo, unsigned char *buf, int bufsize) {
	int rval = X->SetValue(deviceNo, 65441, 4, buf, bufsize) ;
	
	return 1 ;
	}

int 
CUSBaccess::SetLED(int deviceNo, enum LED_IDs Led, int value) {
	unsigned char s[3] ;
	int rval = 0 ;
	
	if (X->GetUSBType(deviceNo) == LED_DEVICE && X->GetVersion(deviceNo) <= 10) {
		s[0] = Led ;
		s[1] = value ;
		rval = SetValue(deviceNo, s, 2) ;
		}
	else {
		s[0] = 0 ;
		s[1] = Led ;
		s[2] = value ;
		rval = SetValue(deviceNo, s, 3) ;
		}

	return rval ;
	}

int 
CUSBaccess::SetSwitch(int deviceNo, enum SWITCH_IDs Switch, int On) {
	unsigned char s[3] ;
	int rval = 0 ;
	
	USBtype_enum devType = X->GetUSBType(deviceNo) ;
	if (devType == SWITCH1_DEVICE || devType == AUTORESET_DEVICE || devType == WATCHDOG_DEVICE) {
		s[0] = 0 ;
		s[1] = Switch ;
		if (X->GetVersion(deviceNo) < 4)	// old version do not invert
			s[2] = !On ;
		else
			s[2] = On ;
		rval = SetValue(deviceNo, s, 3) ;
		if (rval && Switch == SWITCH_0) {			// set LED for first switch
			if (On) {
				SetLED(deviceNo, LED_0, 0) ;	// USB Switch will invert LED
				SetLED(deviceNo, LED_1, 15) ;
				}
			else {
				SetLED(deviceNo, LED_0, 15) ;
				SetLED(deviceNo, LED_1, 0) ;
				}
			}
		}

	return rval ;
	}

int		// 0=error, else=ok 
CUSBaccess::GetSwitchConfig(int deviceNo, int &switchCount, int &buttonAvailable) {
	const int bufSize = 6 ;
	unsigned char buf[bufSize] ;
	int ok = 1 ;
	USBtype_enum devType = X->GetUSBType(deviceNo) ;

	if (devType != SWITCH1_DEVICE && devType != AUTORESET_DEVICE && devType != WATCHDOG_DEVICE)
		return 0 ;

	switchCount = 1 ;
	buttonAvailable = 0 ;
	int version = X->GetVersion(deviceNo) ;
	if (version >= 10) {	
		if (ok = GetValue(deviceNo, buf, bufSize) && (buf[0] & 0x80)) {
			switchCount = 1 ;
			if (buf[0] & 0x02)
				switchCount = 2 ;
			if (buf[0] & 0x08)
				switchCount = 3 ;
			if (buf[0] & 0x20)
				buttonAvailable = 1 ;
			}
		}

	return ok ;
	}

int		// On 0=off, 1=on, -1=error	 
CUSBaccess::GetSwitch(int deviceNo, enum SWITCH_IDs Switch) {
	const int bufSize = 6 ;
	unsigned char buf[bufSize] ;
	int ok = 0 ;
	USBtype_enum devType = X->GetUSBType(deviceNo) ;

	if (devType != SWITCH1_DEVICE && devType != AUTORESET_DEVICE && devType != WATCHDOG_DEVICE)
		return -1 ;

	int version = X->GetVersion(deviceNo) ;
	if (1 || version < 10) {					// else only if in separate thread
		if (GetValue(deviceNo, buf, bufSize)) {
			int mask = 1 << ((Switch - SWITCH_0) * 2) ;
			if (version >= 10)
				ok = (buf[0] & mask) ? 1 : 0 ;
			else	// old switch
				ok = (buf[2] & mask) ? 1 : 0 ;
			}
		else
			ok = -1 ;	// getvalue failed - may be disconnected

		if (ok >= 0 && version < 4)
			ok = !ok ;
		}
	else {		// new version - ask for online count to get a fast answer (use this only if in separate thread)
		static int sequenceNumber = 1 ;

		buf[0] = GetInfo ;
		buf[1] = OnlineCount ;
		buf[2] = sequenceNumber ;
		SetValue(deviceNo, buf, 3) ;
		for (int timeout=25 ; timeout > 0 ; timeout--) {
			Sleep(25) ;
			if (GetValue(deviceNo, buf, bufSize)) {
				if ((buf[0] & 0x80) == 0)	// valid bit
					continue ;
				if (buf[1] != ( (sequenceNumber & 0x1f) << 3 ) + OnlineCount)
					continue ;
				ok = buf[0] & 1 ;
				break ;
				}
			}

		sequenceNumber = (++sequenceNumber) & 0x1f ;
		}

	return ok ;
	}

int 
CUSBaccess::ResetDevice(int deviceNo) {
	int ok = 1 ;
	const int bufsize = 4 ;
	unsigned char buf[bufsize] ;

	buf[0] = CUSBaccess::Reset ;
	buf[1] = 0 ;
	buf[2] = 0 ;
	buf[3] = 0 ;
	if (X->GetUSBType(deviceNo) == TEMPERATURE2_DEVICE)
		ok = SetValue(deviceNo, buf, bufsize) ;
	else
		ok = SetValue(deviceNo, buf, bufsize-1) ;

	return ok ;
	}

int 
CUSBaccess::CalmWatchdog(int deviceNo, int minutes) {
	int ok = 1 ;
	const int bufsize = 3 ;
	unsigned char buf[bufsize] ;

	buf[0] = CUSBaccess::KeepCalm ;
	buf[1] = minutes ;
	buf[2] = 0 ;
	SetValue(deviceNo, buf, bufsize) ;

	return ok ;
	}

int 
CUSBaccess::GetTemperature(int deviceNo, double *Temperature, int *timeID) {
	int ok = 1 ;

	switch (X->GetUSBType(deviceNo)) {
		case TEMPERATURE_DEVICE: {
			const int bufSize = 6 ;
			unsigned char buf[bufSize] ;
			// read temperature 
			if (GetValue(deviceNo, buf, bufSize) == 0) {
				ok = 0 ;
				break ;
				}
			*timeID  = ((buf[0] & 0x7f) << 8) + buf[1] ;
			int value = (buf[2] << 5) + (buf[3] >> 3) ;
			if (value & 0x1000)		// negativ!
				value = (value & 0xfff) - 0x1000 ;
			int valid = (buf[0] & 0x80) ;	// MSB = valid-bit
			if (!valid) { // invalid time
				ok = 0 ;
				break ;
				}
			*Temperature = value * 0.0625 ;
			break ;
			}
		case TEMPERATURE2_DEVICE: {
			const int bufSize = 7 ;
			unsigned char buf[bufSize] ;
			// read temperature 
			if (GetValue(deviceNo, buf, bufSize) == 0) {
				ok = 0 ;
				break ;
				}
			*timeID  = ((buf[0] & 0x7f) << 8) + buf[1] ;
			int value = (buf[2] << 5) + (buf[3] >> 3) ;
			if (value & 0x1000)		// negativ!
				value = (value & 0xfff) - 0x1000 ;
			int valid = (buf[0] & 0x80) ;	// MSB = valid-bit
			if (!valid) { // invalid time
				ok = 0 ;
				break ;
				}
			*Temperature = value * 0.0625 ;
			break ;
			}
		case HUMIDITY1_DEVICE:
		case TEMPERATURE5_DEVICE: {
			const int bufSize = 7 ;
			unsigned char buf[bufSize] ;
			// read temperature 
			if (GetValue(deviceNo, buf, bufSize) == 0) {
				ok = 0 ;
				break ;
				}

			*timeID  = ((buf[0] & 0x7f) << 8) + buf[1] ;
			int humi = (buf[2] << 8) + buf[3] ;
			int temp = (buf[4] << 8) + buf[5] ;
			int value = (buf[2] << 5) + (buf[3] >> 3) ;
			if (value & 0x1000)		// negativ!
				value = (value & 0xfff) - 0x1000 ;
			int valid = (buf[0] & 0x80) ;	// MSB = valid-bit
			if (!valid) { // invalid time
				ok = 0 ;
				break ;
				}
		//	double humidity = -4. + 0.0405 * humi - 2.8 * humi * humi / 1000000 ;
			*Temperature = -40. + 0.01 * temp ;
			break ;
			}
		default:
			ok = 0 ;
			break ;
		}

	return ok ;
	}

int 
CUSBaccess::GetHumidity(int deviceNo, double *Humidity, int *timeID) {
	int ok = 1 ;

	switch (X->GetUSBType(deviceNo)) {
		case HUMIDITY1_DEVICE: {
			const int bufSize = 7 ;
			unsigned char buf[bufSize] ;
			// read temperature 
			if (GetValue(deviceNo, buf, bufSize) == 0) {
				ok = 0 ;
				break ;
				}

			*timeID  = ((buf[0] & 0x7f) << 8) + buf[1] ;
			int humi = (buf[2] << 8) + buf[3] ;
		//	int temp = (buf[4] << 8) + buf[5] ;
			int value = (buf[2] << 5) + (buf[3] >> 3) ;
			if (value & 0x1000)		// negativ!
				value = (value & 0xfff) - 0x1000 ;
			int valid = (buf[0] & 0x80) ;	// MSB = valid-bit
			if (!valid) { // invalid time
				ok = 0 ;
				break ;
				}
		//	*Temperature = -40. + 0.01 * temp ;
			*Humidity = -4. + 0.0405 * humi - 2.8 * humi * humi / 1000000 ;
			break ;
			}
		default:
			ok = 0 ;
			break ;
		}
	return ok ;
	}

